Function GetVideoMode() as Object
    result = {}
    video = CreateObject("roVideoMode") 
    result.mode = video.GetMode() 
    result.next_boot_mode = video.GetModeForNextBoot()
    result.details = video.GetActiveMode()
    result.fps = video.getFPS()
    video = invalid
    return result
End Function
Function SetVideoMode(options as Object) as Boolean
    result = true
    video = CreateObject("roVideoMode") 
    current_mode = video.GetMode() 
    next_boot_mode = video.GetModeForNextBoot()
    if (current_mode <> options.mode) then
        result = result AND video.SetMode(options.mode)
    end If
    if (options.apply_next_boot <> invalid AND options.apply_next_boot AND next_boot_mode <> options.mode) then
        next_boot_result = video.SetModeForNextBoot(options.mode)
        result = result AND next_boot_result
        if next_boot_result <> true then
            video.SetModeForNextBoot("auto")
        end if
    end if
    video = invalid
    return result
End Function
Function ToggleScreen(enabled as Boolean) as Boolean
    video = CreateObject("roVideoMode")
    result = video.SetPowerSaveMode(enabled)
    video = invalid
    return result
End Function
Function SetAudioOutput(options as Object) as Boolean
    result = true
    hdmi = CreateObject("roAudioOutput", "hdmi")
    result = result AND audio_priv_ApplyAudioOutputOptions(hdmi, options)
    hdmi = invalid
    spdif = CreateObject("roAudioOutput", "spdif")
    result = result AND audio_priv_ApplyAudioOutputOptions(spdif, options)
    spdif = invalid
    analog = CreateObject("roAudioOutput", "analog")
    result = result AND audio_priv_ApplyAudioOutputOptions(analog, options)
    analog = invalid
    return result
End Function
Function audio_priv_ApplyAudioOutputOptions(output as Object, options as Object) as Boolean
    result = true
    if (type(output) = "roAudioOutput") then
        if (options.muted <> invalid) then
            result = result AND output.SetMute(options.muted)
        end if
        if (options.volume <> invalid) then
            result = result AND output.SetMute(false)
            result = result AND output.SetVolume(options.volume)
        end if
        if (options.treble <> invalid OR options.bass <> invalid) then
            treble = options.treble
            if (treble = invalid) treble = 0
            bass = options.bass
            if (bass = invalid) bass = 0
            result = result AND output.SetTone(treble, bass)
        end if
    end if
    return result
End Function
Sub InitializeAutoReboot(ty as Object)
    if (type(ty.sysConfig.reboot) = "roAssociativeArray") then
        hour = 0
        if (type(ty.sysConfig.reboot.hour) = "Integer") then hour = ty.sysConfig.reboot.hour
        minute = 0
        if (type(ty.sysConfig.reboot.minute) = "Integer") then minute = ty.sysConfig.reboot.minute
        timer = ty.CreateTimer("autoreboot", autoreboot_OnTimer)
        timer.SetDate(-1, -1, -1)
        timer.SetTime(hour, minute, 0, 0)
        timer.Start()
    end if
End Sub
Sub autoreboot_OnTimer(e as Object)
    RebootSystem()
End Sub
Function CreateFullScreenRectangle() As Object
    res = GetResolution()
    rect = CreateObject("roRectangle", 0, 0, res.width, res.height)
    return rect
End Function
Function GetResolution() As Object
    videoMode = CreateObject("roVideoMode")
    resX = videoMode.GetResX()
    resY = videoMode.GetResY()
    videoMode = invalid
    result = {
        width: resX,
        height: resY,
    }
    return result
End Function
Function IsStringEmptyOrInvalid(value) As Boolean
	if ((type(value) = "roString" OR type(value) = "String") AND value <> "") then 
        return false
    else
        return true
    end if
End Function
Function GetISODateTimeString(date As Object) As String
	iso_date_time$ = date.ToIsoString()
	index = instr(1, iso_date_time$, ",")
	if index >= 1 then
		iso_date_time$ = mid(iso_date_time$, 1, index - 1)
	endif
	return iso_date_time$
End Function
Function GetDirName(file_path as String) as String
    last_slash_index = file_path.Instr("/")
    current_slash_index = last_slash_index
    while current_slash_index >= 0
        last_slash_index = current_slash_index
        current_slash_index = file_path.Instr(current_slash_index + 1, "/")
    end while
    if (last_slash_index >= 0) then
        return file_path.Left(last_slash_index)
    else
        return "/"
    end if
End Function
Sub BubbleSortFileNames(fileNames As Object)
	if type(fileNames) = "roList" then
		n = fileNames.Count()
		while n <> 0
			newn = 0
			for i = 1 to (n - 1)
				if fileNames[i-1] > fileNames[i] then
					k = fileNames[i]
					fileNames[i] = fileNames[i-1]
					fileNames[i-1] = k
					newn = i
				endif
			next
			n = newn
		end while
	endif
End Sub
Function InitSnapshotConfig() as Object
    CreateDirectory("snapshots")
    config = {}
    config.enabled = false
    config.quality = 50
    config.max = 20
    config.list = MatchFiles("/snapshots/", "*.jpg")
    BubbleSortFileNames(config.list)
End Function
Sub TakeSnapshot(systemTime As Object)
    globalAA = GetGlobalAA()
    DeleteExcessSnapshots(globalAA)
    videoMode = CreateObject("roVideoMode")
    dt_local = systemTime.GetLocalDateTime()
    date_time$ = GetISODateTimeString(dt_local)
    options = {}
    options.filename = "snapshots/" + date_time$ + ".jpg"
    options.Width = videoMode.GetResX()
    options.Height = videoMode.GetResY()
    options.filetype = "JPEG"
    options.quality = globalAA.ty.snapshots.quality
    options.Async = 0
    if type(globalAA.ty) = "roAssociativeArray" then
        globalAA.ty.diagnostics.PrintDebug( "------------------------------------------- SCREENSHOT " + options.filename + "-------------------------------------")
    endif
    ok = videoMode.Screenshot(options)
    if not ok then
        if type(globalAA.ty) = "roAssociativeArray" then
            globalAA.ty.diagnostics.PrintDebug("TakeSnapshot: not ok returned from Screenshot")		
        endif
    else
        globalAA.ty.snapshots.list.push(date_time$ + ".jpg")
        if type(globalAA.ty) = "roAssociativeArray" then
            snapshot_captured = CreateObject("roAssociativeArray")
            snapshot_captured["EventType"] = "SNAPSHOT_CAPTURED"
            snapshot_captured["SnapshotName"] = date_time$ + ".jpg"
            globalAA.ty.msgPort.PostMessage(snapshot_captured)			
        endif
    endif
    videoMode = invalid
End Sub
Sub DeleteExcessSnapshots(globalAA As Object)
	while globalAA.ty.snapshots.list.Count() >= globalAA.ty.snapshots.max and globalAA.ty.snapshots.list.Count() > 0
		file_to_delete = globalAA.ty.snapshots.list.Shift()
		ok = DeleteFile("snapshots/" + file_to_delete)
	end while
End Sub
Sub DisplayErrorScreen(msg1$ As String, msg2$ As String, msToWaitBeforeRebooting As Integer)
    resolution = GetResolution()
    text_params = CreateObject("roAssociativeArray")
    text_params.LineCount = 1
    text_params.TextMode = 2
    text_params.Rotation = 0
    text_params.Alignment = 1
    center_y = resolution.height / 2
    line_height = resolution.height / 32
    rect_line1 = CreateObject("roRectangle", 0, center_y - line_height, resolution.width, line_height)
    text_line1 = CreateObject("roTextWidget", rect_line1, 1, 2, text_params)
    text_line1.PushString(msg1$)
    text_line1.Show()
    rect_line2 = CreateObject("roRectangle", 0, center_y, resolution.width, line_height)
    text_line2 = CreateObject("roTextWidget", rect_line2, 1, 2, text_params)
    text_line2.PushString(msg2$)
    text_line2.Show()
    globalAA = GetGlobalAA()
    if type(globalAA.ty) = "roAssociativeArray" then
        if type(globalAA.ty.msgPort) = "roMessagePort" then
            globalAA.ty.msgPort.SetWatchdogTimeout(0)
        endif
        if (type(globalAA.ty.snapshots) = "roAssociativeArray" AND globalAA.ty.snapshots.enabled) then
            Sleep(1000)	' sleep here ensures that graphics makes it to the screen before the snapshot is taken
            TakeSnapshot(globalAA.ty.systemTime)
        endif
    endif
    wait_port = CreateObject("roMessagePort")
    wait_msg = Wait(msToWaitBeforeRebooting, wait_port)
    RebootSystem()
End Sub
Sub InitializeAutoUpdate(ty as Object)
    hour = 2
    minute = 0
    if (type(ty.sysConfig.update) = "roAssociativeArray") then
        if (type(ty.sysConfig.update.hour) = "Integer") then hour = ty.sysConfig.update.hour
        if (type(ty.sysConfig.update.minute) = "Integer") then minute = ty.sysConfig.update.minute
    end if
    timer = ty.CreateTimer("autoupdate", autoupdate_OnTimer)
    timer.SetDate(-1, -1, -1)
    timer.SetTime(hour, minute, 0, 0)
    timer.Start()
    if (upd_priv_ShouldUpdateImmediately()) then
        SearchForUpdate()
    end if
End Sub
Sub autoupdate_OnTimer(e as Object)
    SearchForUpdate()
End Sub
Sub SearchForUpdate()
    upd_priv_DownloadSpec()
End Sub
Sub upd_priv_DownloadSpec()
    CreateDirectory("update")
    ty = GetGlobalAA().ty
    transfer = CreateObject("roUrlTransfer")
    ty.AddToEventLoop(transfer, "roUrlEvent", Function (e as Object) as Boolean
        if (e.GetInt() <> 1) then return false
        if (e.GetResponseCode() = 200) then
            upd_priv_WriteLastUpdateCheck()
            upd_priv_CheckSpec()
        else
            ty = GetGlobalAA().ty
            ty.diagnostics.PrintDebug("update error: download spec failed with code " + e.GetResponseCode().toStr())
        end if
        return true
    End Function)
    url = "https://touchifyapplications.blob.core.windows.net/brightsign"
    if (ty.appConfig.id.Instr(".beta") <> -1) then url = url + "/beta" 
    url = url + "/sync.json"
    transfer.SetUrl(url)
    transfer.AsyncGetToFile("update/sync.json")
End Sub
Sub upd_priv_CheckSpec()
    current_spec = CreateObject("roSyncSpec")
    if (current_spec.ReadFromFile("sync.json")) then
        sync_spec = CreateObject("roSyncSpec")
        if (sync_spec.ReadFromFile("update/sync.json")) then
            if (NOT current_spec.EqualTo(sync_spec)) then
                current_spec = invalid
                upd_priv_SyncUpdate(sync_spec)
            end if
        else
            ty = GetGlobalAA().ty
            ty.diagnostics.PrintDebug("update error: invalid remote sync spec: " + sync_spec.GetFailureReason())
        end if
    else
        current_spec = invalid
    end if
End Sub
Sub upd_priv_SyncUpdate(sync_spec as Object)
    CreateDirectory("update/pool")
    ty = GetGlobalAA().ty
    asset_pool = CreateObject("roAssetPool", "update/pool")
    asset_collection = sync_spec.GetAssets("download")
    if (asset_pool.AssetsReady(asset_collection)) then
        upd_priv_ApplyUpdate(asset_pool, asset_collection, sync_spec)
    else
        asset_fetcher = CreateObject("roAssetFetcher", asset_pool)
        user_data = {
            asset_pool: asset_pool,
            asset_collection: asset_collection,
            sync_spec: sync_spec
        }
        ty.AddToEventLoopWithUserData(asset_fetcher, "roAssetFetcherEvent", user_data, Function (e as Object) as Boolean
            event_code = e.GetEvent()
            if (event_code = 1 OR event_code = -1) then return false
            if (event_code = -2) then 
                ty = GetGlobalAA().ty
                ty.diagnostics.PrintDebug("update error: download error: " + e.GetFailureReason())
            end if
            if (event_code = 2) then
                user_data = e.GetUserData()
                upd_priv_ApplyUpdate(user_data.asset_pool, user_data.asset_collection, user_data.sync_spec)
            end if
            return true
        End Function)
        asset_fetcher.AsyncDownload(asset_collection)
    end if
End Sub
Sub upd_priv_ApplyUpdate(asset_pool as Object, asset_collection as Object, sync_spec as Object)
    if (NOT asset_pool.AssetsReady(asset_collection)) then
        ty = GetGlobalAA().ty
        ty.diagnostics.PrintDebug("update error: assets not ready!")
        return
    end if
    if (NOT asset_pool.Validate(sync_spec, { DeleteCorrupt: true })) then 
        ty = GetGlobalAA().ty
        ty.diagnostics.PrintDebug("update error: some assets are invalid: " + asset_pool.GetFailureReason())
        return
    end if
    asset_pool_files = CreateObject("roAssetPoolFiles", asset_pool, asset_collection)
    files = asset_collection.GetAssetList()
    for each file_info in files
        src_file_path = asset_pool_files.GetPoolFilePath(file_info.name)
        dest_file_path = "/" + file_info.name
        dirname = GetDirName(dest_file_path)
        if (dirname <> "/") then CreateDirectory(dirname)
        if (NOT CopyFile(src_file_path, dest_file_path)) then
            upd_priv_RetryApplyUpdate(asset_pool, asset_collection, sync_spec)
            return
        end if
    end for
    sync_spec.WriteToFile("sync.json", { format: "json" })
    RebootSystem()
End Sub
Sub upd_priv_RetryApplyUpdate(asset_pool as Object, asset_collection as Object, sync_spec as Object)
    ty = GetGlobalAA().ty
    if (ty.temps.update_retried <> true) then
        ty.temps.update_retried = true
        upd_priv_ApplyUpdate(asset_pool, asset_collection, sync_spec)
    else
        ty.html.Hide()
        DisplayErrorScreen("Update error!", "An error occured while trying to apply Touchify update. Please contact the support.", 0)
    end if
End Sub
Function upd_priv_ShouldUpdateImmediately() as Boolean
    iso_date = ReadAsciiFile("update/last_check.txt")
    if (iso_date = invalid) then return true
    date = CreateObject("roDateTime")
    if (NOT date.FromIsoString(iso_date)) then return true
    ty = GetGlobalAA().ty
    update_sec = date.ToSecondsSinceEpoch()
    current_sec = ty.systemTime.GetUtcDateTime().ToSecondsSinceEpoch()
    last_update_sec = current_sec - update_sec
    day_sec = 24 * 3600
    return last_update_sec >= day_sec
End Function
Sub upd_priv_WriteLastUpdateCheck()
    ty = GetGlobalAA().ty
    iso_date = ty.systemTime.GetUtcDateTime().ToIsoString()
    WriteAsciiFile("update/last_check.txt", iso_date)
End Sub
Function GetAppConfig() as Object 
    config = config_priv_ReadAppConfig("config.json")
    if config <> invalid then
        return config_priv_ApplyLocalConfig(config)
    else
        DisplayErrorScreen("This device has no configuration available", "Missing config.json file in your storage", 0)
    end if
End Function
Function SaveAppConfigSection(section as String, value as Object) as Boolean
    config = config_priv_ReadAppConfig("config.local.json")
    if (type(config.brightsign) <> "roAssociativeArray") then
        config.brightsign = {}
    end if
    config.brightsign[section] = value
    config_json = FormatJson(config)
    config_bytes = CreateObject("roByteArray")
    config_bytes.FromAsciiString(config_json)
    result = config_bytes.WriteFile("config.local.json")
    return result
End Function
Function config_priv_ApplyLocalConfig(config as Object) as Object
    config_local = config_priv_ReadAppConfig("config.local.json")
    if (config_local <> invalid) then
        for each config_local_key in config_local
            config[config_local_key] = config_local[config_local_key]
        end for
    end if
    return config
End Function
Function config_priv_ReadAppConfig(path as String) as Object 
    data = CreateObject("roByteArray")
    success = data.ReadFile(path)
    if (success AND data.Count() > 0) then
        config_json = data.ToAsciiString()  
        config = ParseJson(config_json)
        return config
    else
        return invalid
    end if
End Function
Function NewDiagnostics(sysConfig As Object) As Object
    diagnostics = CreateObject("roAssociativeArray")
    diagnostics.debug = false
    diagnostics.systemLogDebug = false
    if (type(sysConfig.diagnostics) = "roAssociativeArray") then
        diagnostics.debug = sysConfig.diagnostics.debug
        diagnostics.systemLogDebug = sysConfig.diagnostics.systemLogDebug
    end if
    diagnostics.autorunVersion$ = "unknown"
    diagnostics.customAutorunVersion$ = "unknown"
    diagnostics.firmwareVersion$ = "unknown"
    diagnostics.systemTime = CreateObject("roSystemTime")
    if diagnostics.systemLogDebug then
		diagnostics.systemLog = CreateObject("roSystemLog")
	endif
    diagnostics.UpdateDebugOn = diag_UpdateDebugOn
    diagnostics.UpdateSystemLogDebugOn = diag_UpdateSystemLogDebugOn
    diagnostics.PrintDebug = diag_PrintDebug
    diagnostics.PrintTimestamp = diag_PrintTimestamp
    diagnostics.SetSystemInfo = diag_SetSystemInfo
    return diagnostics
End Function
Sub diag_UpdateDebugOn(debugOn As Boolean)
    m.debug = debugOn
End Sub
Sub diag_UpdateSystemLogDebugOn(systemLogDebug As Boolean)
    m.systemLogDebug = systemLogDebug
    if systemLogDebug and type(m.systemLog) <> "roSystemLog" then
		m.systemLog = CreateObject("roSystemLog")
    endif
End Sub
Sub diag_PrintDebug(debugStr$ As String)
    if type(m) <> "roAssociativeArray" then stop
    if m.debug then 
        print debugStr$
    endif
    if m.systemLogDebug then
		m.systemLog.SendLine(debugStr$)
	endif
    return
End Sub
Sub diag_PrintTimestamp()
    eventDateTime = m.systemTime.GetLocalDateTime()
    if m.debug then print eventDateTime.GetString()
    if m.systemLogDebug then
		m.systemLog.SendLine(eventDateTime.GetString())
	endif
    return
End Sub
Sub diag_SetSystemInfo(sysInfo As Object, diagnosticCodes As Object)
    m.autorunVersion$ = sysInfo.autorunVersion$
    m.customAutorunVersion$ = sysInfo.customAutorunVersion$
    m.firmwareVersion$ = sysInfo.deviceFWVersion$
    m.deviceUniqueID$ = sysInfo.deviceUniqueID$
    m.deviceModel$ = sysInfo.deviceModel$
    m.deviceFamily$ = sysInfo.deviceFamily$
    m.modelSupportsWifi = sysInfo.modelSupportsWifi
    m.enableLogDeletion = sysInfo.enableLogDeletion
    m.diagnosticCodes = diagnosticCodes
    return
End Sub
Sub InitializeHtmlWidget(ty as Object)
    rect = CreateFullScreenRectangle()
    html_config = CreateHtmlConfig(ty.msgPort, ty.appConfig)
    html = CreateObject("roHtmlWidget", rect, html_config)
    ty.html = html
    ty.handlers["roHtmlWidgetEvent"] = HandleHtmlEvent
    html.Show()
End Sub
Sub HandleHtmlEvent(event as Object)
End Sub
Function CreateHtmlConfig(msgPort as Object, appConfig as Object) as Object
    CreateDirectory("data")
    config = {
        url: "file:///SD:/index.html",
        port: msgPort,
        nodejs_enabled: true,
        javascript_enabled: true,
        brightsign_js_objects_enabled: true,
        mouse_enabled: true,
        hwz_default: "z-index:-1",
        storage_path: "data",
        storage_quota: GetHtmlStorageQuota(),
        security_params: {
            websecurity: false,
            camera_enabled: true
        }
    }
    if (appConfig.devTools <> invalid AND appConfig.devTools) then
        config.inspector_server = { port: 2016 }
    end if
    if (type(appConfig.window) = "roAssociativeArray") then
        if (NOT IsStringEmptyOrInvalid(appConfig.window.transform)) then
            config.transform = appConfig.window.transform
        end if
    end if
    return config
End Function
Function GetHtmlStorageQuota() as Double
    info = CreateObject("roStorageInfo", "SD:")
    size_mb# = info.GetSizeInMegabytes()
    size# = size_mb# * 1024 * 1024
    quota = size# / 4
    return quota
End Function
Function GetNetworkInfo(wiredOrWifi% as Integer) as Object
    nc = CreateObject("roNetworkConfiguration", wiredOrWifi%)
    result = net_priv_GetNetworkConfiguration(nc, wiredOrWifi%)
    nc = invalid
    return result
End Function
Function SetNetworkInfo(info as Object)
    wired_config = CreateObject("roNetworkConfiguration", 0)
    wired = net_priv_GetNetworkConfiguration(wired_config, 0)
    wifi_config = CreateObject("roNetworkConfiguration", 1)
    wifi = net_priv_GetNetworkConfiguration(wifi_config, 1)
    result = True
    net_config = {}
    if (net_priv_HasNetworkConfigurationChanged(wired, wifi, info)) then
        if (wired_config <> invalid) then wired_config.ResetInterfaceSettings()
        if (wifi_config <> invalid) then wifi_config.ResetInterfaceSettings()
        if (info.type = "wired") then
            net_config = wired_config
        else
            net_config = wifi_config
            if (wifi_config = invalid) then
                DisplayErrorScreen("No WiFi interface", "There is no WiFi interface on this device. Please configure the wired network.", 0)
            end if
            result = result AND net_config.SetWiFiESSID(info.wifi_essid)
            if (NOT IsStringEmptyOrInvalid(info.wifi_password)) then
                result = result AND net_config.SetWiFiPassphrase(info.wifi_password)
            end if
            if (NOT IsStringEmptyOrInvalid(info.wifi_identity)) then
                result = result AND net_config.SetWiFiIdentity(info.wifi_identity)
            end if
            if (NOT IsStringEmptyOrInvalid(info.wifi_eap_tls_options)) then
                result = result AND net_config.SetWiFiEapTlsOptions(info.wifi_eap_tls_options)
            end if
            if (NOT IsStringEmptyOrInvalid(info.wifi_eap_tls_options)) then
                result = result AND net_config.SetWiFiEapTlsOptions(info.wifi_eap_tls_options)
            end if
            if (NOT IsStringEmptyOrInvalid(info.wifi_ca_certificate)) then
                result = result AND net_config.SetWiFiCACertificates(info.wifi_ca_certificate)
            end if
            if (NOT IsStringEmptyOrInvalid(info.wifi_client_certificate)) then
                result = result AND net_config.SetWiFiClientCertificate(info.wifi_client_certificate)
            end if
            if (NOT IsStringEmptyOrInvalid(info.wifi_private_key)) then
                result = result AND net_config.SetWiFiPrivateKey(info.wifi_private_key)
            end if
            if (NOT IsStringEmptyOrInvalid(info.wifi_security_mode)) then
                result = result AND net_config.SetWiFiSecurityMode(info.wifi_security_mode)
            end if
        end if
        if info.dhcp then
            result = result AND net_config.SetDHCP()
        else
            result = result AND net_config.SetIP4Address(info.ip4_address)
            if (NOT IsStringEmptyOrInvalid(info.ip4_netmask)) then
                result = result AND net_config.SetIP4Netmask(info.ip4_netmask)
            end if
            if (NOT IsStringEmptyOrInvalid(info.ip4_gateway)) then
                result = result AND net_config.SetIP4Gateway(info.ip4_gateway)
            end if
            if (NOT IsStringEmptyOrInvalid(info.ip4_broadcast)) then
                result = result AND net_config.SetIP4Broadcast(info.ip4_broadcast)
            end if
            if (info.dns_servers <> invalid) then
                result = result AND net_config.SetDNSServers(info.dns_servers)
            end if
        end if
        net_config.ResetHostSettings()
        if (NOT IsStringEmptyOrInvalid(info.domain)) then
            result = result AND net_config.SetDomain(info.domain)
        end if
        if (NOT IsStringEmptyOrInvalid(info.hostname)) then
            result = result AND net_config.SetHostName(info.hostname)
        end if
        if (NOT IsStringEmptyOrInvalid(info.configured_proxy)) then
            result = result AND net_config.SetProxy(info.configured_proxy)
            result = result AND net_config.SetProxyBypass(["localhost"])
        end if 
        if (NOT IsStringEmptyOrInvalid(info.time_server)) then
            result = result AND net_config.SetTimeServer(info.time_server)
        else
            result = result AND net_config.SetTimeServer("http://time.brightsignnetwork.com")
        end if
        result = result AND net_config.Apply()
    end if
    net_config = invalid
    wired_config = invalid
    wifi_config = invalid
    return result
End Function
Function GetSSHInfo() as Object
    result = {}
    registry = CreateObject("roRegistrySection", "networking")
    port = registry.Read("ssh")
    if (port <> "") then
        result.enabled = true
        result.port = port
    else
        result.enabled = false
        result.port = "22"
    end if
    registry = invalid
    return result
End Function
Function SetSSHInfo(info as Object) as Boolean
    if (info.port = invalid) then info.port = "22"
    registry = CreateObject("roRegistrySection", "networking")
    current_port = registry.Read("ssh")
    current_enabled = current_port <> ""
    changed = info.enabled <> current_enabled OR current_port <> info.port
    result = true
    if (changed) then
        if (info.enabled) then
            result = result AND registry.Write("ssh", info.port)
        else
            result = result AND registry.Delete("ssh")
        end if
        result = result AND registry.Flush()
    end if
    registry = invalid
    if (info.enabled AND info.password <> invalid) then
        net_conf = CreateObject("roNetworkConfiguration", 0)
        result = result AND net_conf.SetLoginPassword(info.password)
        result = result AND net_conf.Apply()
        net_conf = invalid
    end if
    if (changed) then
        RebootSystem()
    end if
    return result
End Function
Function SetDWSConfig(config as Object) as Boolean
    result = true
    reboot_required = false
    dws_config = {}
    if (config.enabled <> invalid AND config.enabled) then
        dws_config.port = "default"
        if (config.port <> invalid) then dws_config.port = config.port
        if (config.password <> invalid) then dws_config.open = config.password
        if (config.basic <> invalid) then dws_config.basic = config.basic
    else
        dws_config.port = "0"
    end if
    network_config = CreateObject("roNetworkConfiguration", 0)
    if type(network_config) = "roNetworkConfiguration"
        reboot_required = network_config.SetupDWS(dws_config)
        if (NOT reboot_required) then result = network_config.GetFailureReason() = ""
    endif
    network_config = invalid
    if (reboot_required) then
        RebootSystem()
    end if
    return result
End Function
Function net_priv_GetNetworkConfiguration(nc as Object, wiredOrWifi% as Integer) as Object
    cc = invalid
    if (type(nc) = "roNetworkConfiguration") then cc = nc.GetCurrentConfig()
    result = {}
    if (cc <> invalid) then
        result.enabled = false
        result.type = cc.type
        result.metric = cc.metric
        result.hostname = cc.hostname
        result.ip4_address = cc.ip4_address
        result.ip4_netmask = cc.ip4_netmask
        result.ip4_gateway = cc.ip4_gateway
        result.ip4_broadcast = cc.ip4_broadcast
        result.ethernet_mac = cc.ethernet_mac
        result.domain = cc.domain
        result.dhcp = cc.dhcp
        result.dns_servers = cc.dns_servers
        result.time_server = cc.time_server
        result.configured_proxy = cc.configured_proxy
        if wiredOrWifi% = 1 then
            result.wifi_essid = cc.wifi_essid
            result.enabled = NOT IsStringEmptyOrInvalid(cc.wifi_essid) AND (cc.dhcp OR NOT IsStringEmptyOrInvalid(cc.ip4_address))
        else
            result.enabled = cc.dhcp OR NOT IsStringEmptyOrInvalid(cc.ip4_address)
        end if
    else
        result.enabled = false
        result.metric = -1
        result.hostname = ""
        result.ip4_address = ""
        result.ip4_netmask = ""
        result.ip4_gateway = ""
        result.ip4_broadcast = ""
        result.ethernet_mac = ""
        result.domain = ""
        result.dhcp = true
        result.dns_servers = []
        result.time_server = ""
        result.configured_proxy = ""
        if wiredOrWifi% = 1 then
            result.type = "wifi"
            result.wifi_essid = ""
        else
            result.type = "wired"
        end if
    end if
    cc = invalid
    return result
End Function
Function net_priv_HasNetworkConfigurationChanged(wired as Object, wifi as Object, info as Object) as Boolean
    config = {}
    if (info.force <> invalid AND info.force) then
        return true
    end if
    if (info.type = "wired") then
        if (wifi.enabled) then return True
        config = wired
    else
        if (wired.enabled OR wifi.wifi_essid <> info.wifi_essid) then return True
        config = wifi
    end if
    if (config.dhcp <> info.dhcp) then
        return True
    else if (NOT config.dhcp) then
        if (config.ip4_address <> info.ip4_address) then
            return True
        else if (config.ip4_netmask <> info.ip4_netmask) then
            return True
        else if (config.ip4_gateway <> info.ip4_gateway) then
            return True
        else if (config.ip4_broadcast <> info.ip4_broadcast) then
            return True
        else if (net_priv_HasDnsChanged(config.dns_servers, info.dns_servers)) then
            return True
        end if
    else if (config.hostname <> info.hostname) then
        return True
    else if (config.domain <> info.domain) then
        return True
    else if (config.configured_proxy <> info.configured_proxy) then
        return True
    else if (NOT IsStringEmptyOrInvalid(info.time_server) AND config.time_server <> info.time_server) then
        return True
    end if
    return False
End Function
Function net_priv_HasDnsChanged(src_dns as Object, dest_dns as Object) as Boolean
    if (dest_dns = invalid) return False
    if (src_dns.Count() <> dest_dns.Count()) then return True
    For i=0 To src_dns.Count() - 1 Step 1
        if (src_dns[i] <> dest_dns[i]) then return True
    End For
    return False
End Function
Function GetTimeZone() As Object
    time = CreateObject("roSystemTime")
    result = {}
    result.zone = time.GetTimeZone()
    time = invalid
    return result
End Function
Function SetTimeZone(zone As String) As Boolean
    time = CreateObject("roSystemTime")
    result = time.SetTimeZone(zone)
    time = invalid
    return result
End Function
Function GetStorageStatus() as Object
    info = CreateObject("roStorageInfo", "SD:")
    result = {}
    result.total = info.GetSizeInMegabytes()
    result.used = info.GetUsedInMegabytes()
    result.free = info.GetFreeInMegabytes()
    result.type = info.GetFileSystemType()
    info = invalid
    return result
End Function
Function GetDeviceInfo() as Object
    result = {}
    device = CreateObject("roDeviceInfo")
    result.model = device.GetModel()
    result.version = device.GetVersion()
    result.serial = device.GetDeviceUniqueId()
    device = invalid
    network = CreateObject("roNetworkConfiguration", 0)
    network_config = network.GetCurrentConfig()
    result.hostname = network_config.hostname
    network_config = invalid
    network = invalid
    return result
End Function
Function GetInputDevices() as Object
    result = {}
    touchscreen = CreateObject("roTouchScreen")
    result.touch = touchscreen.GetDeviceName() <> ""
    result.mouse = touchscreen.IsMousePresent()
    touchscreen = invalid
    keyboard = CreateObject("roKeyboard")
    result.keyboard = keyboard.IsPresent()
    keyboard = invalid
    return result
End Function
Function InitializeHttpServer(ty as Object) as Object
    server = CreateObject("roHttpServer", { port: 1990 })
    server.SetPort(ty.msgPort)
    server.AddGetFromEvent({ url_path: "/config", user_data: http_GetAppConfig, passwords: invalid })
    server.AddGetFromEvent({ url_path: "/network", user_data: http_GetNetworkInfo, passwords: invalid })
    server.AddPostToString({ url_path: "/network", user_data: http_PostNetworkInfo, passwords: invalid })
    server.AddGetFromEvent({ url_path: "/ssh", user_data: http_GetSSHInfo, passwords: invalid })
    server.AddPostToString({ url_path: "/ssh", user_data: http_PostSSHInfo, passwords: invalid })
    server.AddPostToString({ url_path: "/dws", user_data: http_PostDwsConfig, passwords: invalid })
    server.AddGetFromEvent({ url_path: "/video-mode", user_data: http_GetVideoMode, passwords: invalid })
    server.AddPostToString({ url_path: "/video-mode", user_data: http_PostVideoMode, passwords: invalid })
    server.AddPostToString({ url_path: "/audio-output", user_data: http_PostAudioOutput, passwords: invalid })
    server.AddPostToString({ url_path: "/toggle-screen", user_data: http_PostToggleScreen, passwords: invalid })
    server.AddGetFromEvent({ url_path: "/device", user_data: http_GetDeviceInfo, passwords: invalid })
    server.AddPostToString({ url_path: "/device/reboot", user_data: http_PostReboot, passwords: invalid })
    server.AddPostToString({ url_path: "/device/restart", user_data: http_PostRestart, passwords: invalid })
    server.AddGetFromEvent({ url_path: "/time-zone", user_data: http_GetTimeZone, passwords: invalid })
    server.AddPostToString({ url_path: "/time-zone", user_data: http_PostTimeZone, passwords: invalid })
    server.AddGetFromEvent({ url_path: "/storage", user_data: http_GetStorageStatus, passwords: invalid })
    server.AddGetFromEvent({ url_path: "/inputs", user_data: http_GetInputsContext, passwords: invalid })
    server.AddPostToString({ url_path: "/update", user_data: http_PostUpdate, passwords: invalid })
    ty.server = server
    ty.handlers["roHttpEvent"] = HandleHttpEvent
    return server
End Function
Sub HandleHttpEvent(event as Object)
    handler = event.GetUserData()
    if (type(handler) = "roFunction") then
        handler(event)
    end if
End Sub
Sub http_GetAppConfig(e as Object)
    ty = GetGlobalAA().ty
    config = ty.appConfig
    result = CreateObject("roAssociativeArray")
    result.SetModeCaseSensitive()
    for each config_key in config
        if (config_key <> "brightsign") then
            result[config_key] = config[config_key]
        end if
    end for
    SendHttpResponse(e, result, 200)
End Sub
Sub http_GetNetworkInfo(e as Object)
    result = {}
    result.wired = GetNetworkInfo(0)
    result.wifi = GetNetworkInfo(1)
    SendHttpResponse(e, result, 200)
End Sub
Sub http_PostNetworkInfo(e as Object)
    request = ParseJson(e.GetRequestBodyString())
    success = SetNetworkInfo(request)
    success = success AND SaveAppConfigSection("network", request)
    if (success) then
        SendEmptyHttpResponse(e, 204)
    else
        result = {}
        result.code = "INVALID_NETWORK_CONFIG"
        result.message = "invalid network configuration"
        SendHttpResponse(e, result, 400)
    end if
End Sub
Sub http_GetSSHInfo(e as Object)
    result = GetSSHInfo()
    SendHttpResponse(e, result, 200)
End Sub
Sub http_PostSSHInfo(e as Object)
    request = ParseJson(e.GetRequestBodyString())
    success = SetSSHInfo(request)
    success = success AND SaveAppConfigSection("ssh", request)
    if (success) then
        SendEmptyHttpResponse(e, 204)
    else
        result = {}
        result.code = "INVALID_SSH_CONFIG"
        result.message = "invalid SSH configuration"
        SendHttpResponse(e, result, 400)
    end if
End Sub
Sub http_PostDwsConfig(e as Object)
    request = ParseJson(e.GetRequestBodyString())
    success = SetDWSConfig(request)
    success = success AND SaveAppConfigSection("dws", request)
    if (success) then
        SendEmptyHttpResponse(e, 204)
    else
        result = {}
        result.code = "INVALID_DWS_CONFIG"
        result.message = "invalid Diagnostics Web Console configuration"
        SendHttpResponse(e, result, 400)
    end if
End Sub
Sub http_GetVideoMode(e as Object)
    result = GetVideoMode()
    SendHttpResponse(e, result, 200)
End Sub
Sub http_PostVideoMode(e as Object)
    request = ParseJson(e.GetRequestBodyString())
    success = SetVideoMode(request)
    if (success AND request.apply_next_boot <> invalid AND request.apply_next_boot) then
        conf_section = { mode: request.mode }
        success = success AND SaveAppConfigSection("video", conf_section)
    end if
    if (success) then
        SendEmptyHttpResponse(e, 204)
    else
        result = {}
        result.code = "INVALID_VIDEO_MODE"
        result.message = "invalid video configuration"
        SendHttpResponse(e, result, 400)
    end if
End Sub
Sub http_PostAudioOutput(e as Object)
    request = ParseJson(e.GetRequestBodyString())
    success = SetAudioOutput(request)
    success = success AND SaveAppConfigSection("audio", request)
    if (success) then
        SendEmptyHttpResponse(e, 204)
    else
        result = {}
        result.code = "INVALID_AUDIO_MODE"
        result.message = "invalid audio configuration"
        SendHttpResponse(e, result, 400)
    end if
End Sub
Sub http_PostToggleScreen(e as Object)
    request = ParseJson(e.GetRequestBodyString())
    success = ToggleScreen(request.enabled)
    if (success) then
        SendEmptyHttpResponse(e, 204)
    else
        result = {}
        result.code = "SCREEN_ERROR"
        result.message = "screen did not toggle."
        SendHttpResponse(e, result, 500)
    end if
End Sub
Sub http_GetDeviceInfo(e as Object)
    result = GetDeviceInfo()
    SendHttpResponse(e, result, 200)
End Sub
Sub http_PostReboot(e as Object)
    RebootSystem()
    SendEmptyHttpResponse(e, 204)
End Sub
Sub http_PostRestart(e as Object)
    RestartScript()
    SendEmptyHttpResponse(e, 204)
End Sub
Sub http_GetTimeZone(e as Object)
    result = GetTimeZone()
    SendHttpResponse(e, result, 200)
End Sub
Sub http_PostTimeZone(e as Object)
    request = ParseJson(e.GetRequestBodyString())
    success = SetTimeZone(request.zone)
    success = success AND SaveAppConfigSection("system", request)
    if (success) then
        SendEmptyHttpResponse(e, 204)
    else
        result = {}
        result.code = "INVALID_TIMEZONE_MODE"
        result.message = "invalid time zone configuration"
        SendHttpResponse(e, result, 400)
    end if
End Sub
Sub http_GetStorageStatus(e as Object)
    result = GetStorageStatus()
    SendHttpResponse(e, result, 200)
End Sub
Sub http_GetInputsContext(e as Object)
    result = GetInputDevices()
    SendHttpResponse(e, result, 200)
End Sub
Sub http_PostUpdate(e as Object)
    SearchForUpdate()
    SendEmptyHttpResponse(e, 204)
End Sub
Sub SendHttpResponse(e as Object, result, statusCode as Integer)
    if type(result) = "roAssociativeArray" then
        ret$ = FormatJson(result)
    else
        ret$ = result
    end if
    e.SetResponseBodyString(ret$)
    e.SendResponse(statusCode)
End Sub
Sub SendEmptyHttpResponse(e as Object, statusCode as Integer)
    e.SendResponse(statusCode)
End Sub
Sub InitializeTimersApi(ty as Object)
    ty.timers = CreateObject("roAssociativeArray")
    ty.timers.SetModeCaseSensitive()
    ty.CreateTimer = ty_CreateTimer
    ty.CancelTimer = ty_CancelTimer
    ty.handlers["roTimerEvent"] = HandleTimersEvent
End Sub
Function ty_CreateTimer(name as String, handler as Object) as Object
    timer = CreateObject("roTimer")
    timer.SetPort(m.msgPort)
    timer.SetUserData(handler)
    m.timers[name] = timer
    return timer
End Function
Sub ty_CancelTimer(name as String)
    timer = m.timers[name]
    if (type(timer) = "roTimer") then
        timer.Stop()
        m.timers[name] = invalid
    end if
End Sub
Sub HandleTimersEvent(event as Object)
    handler = event.GetUserData()
    if (type(handler) = "roFunction") then
        handler(event)
    end if
End Sub
Function NewTouchify(msgPort as Object) as Object
    ty = {}
    globalAA = GetGlobalAA()
    globalAA.ty = ty
    ty.global = globalAA
    ty.msgPort = msgPort
    app_config = GetAppConfig()
    ty.appConfig = app_config
    sys_config = {}
    if (type(app_config.brightsign) = "roAssociativeArray") then sys_config = app_config.brightsign
    ty.sysConfig = sys_config
    ty.systemTime = CreateObject("roSystemTime")
    ty.diagnostics = NewDiagnostics(sys_config)
    ty.snapshots = InitSnapshotConfig()
    ty.handlers = CreateObject("roAssociativeArray")
    ty.handlers.SetModeCaseSensitive()
    ty.locks = CreateObject("roAssociativeArray")
    ty.locks.SetModeCaseSensitive()
    InitializeTimersApi(ty)
    ty.InitializeSystem = ty_InitializeSystem
    ty.InitializeServices = ty_InitializeServices
    ty.InitializeWebView = ty_InitializeWebView
    ty.EventLoop = ty_EventLoop
    ty.AddToEventLoop = ty_AddToEventLoop
    ty.AddToEventLoopWithUserData = ty_AddToEventLoopWithUserData
    return ty
End Function
Sub ty_InitializeSystem()
    if (type(m.sysConfig.network) = "roAssociativeArray") then
        if (NOT SetNetworkInfo(m.sysConfig.network)) then
            DisplayErrorScreen("Network configuration error!", "An error occured while trying to setup the network. Please review your configuration file.", 0)
        end if
    end if
    if (type(m.sysConfig.ssh) = "roAssociativeArray") then
        if (NOT SetSSHInfo(m.sysConfig.ssh)) then
            DisplayErrorScreen("SSH configuration error!", "An error occured while trying to setup SSH. Please review your configuration file.", 0)
        end if
    end if
    if (type(m.sysConfig.dws) = "roAssociativeArray") then
        if (NOT SetDWSConfig(m.sysConfig.dws)) then
            DisplayErrorScreen("DWS configuration error!", "An error occured while trying to setup the Diagnostic Web Console. Please review your configuration file.", 0)
        end if
    end if
    if (type(m.sysConfig.audio) = "roAssociativeArray") then
        if (NOT SetAudioOutput(m.sysConfig.audio)) then
            DisplayErrorScreen("Audio configuration error!", "An error occured while trying to setup audio output. Please review your configuration file.", 0)
        end if
    end if
    if (type(m.sysConfig.video) = "roAssociativeArray") then
        video_conf = { mode: m.sysConfig.video.mode, apply_next_boot: true }
        if (NOT SetVideoMode(video_conf)) then
            DisplayErrorScreen("Video configuration error!", "An error occured while trying to setup video mode. Please review your configuration file.", 0)
        end if
    end if
    if (type(m.sysConfig.system) = "roAssociativeArray") then
        if (m.sysConfig.system.zone <> invalid) then
            if (NOT SetTimeZone(m.sysConfig.system.zone)) then
                DisplayErrorScreen("TimeZone configuration error!", "An error occured while trying to setup the current time zone. Please review your configuration file.", 0)
            end if
        end if
    end if
End Sub
Sub ty_InitializeServices()
    InitializeAutoReboot(m)
    InitializeAutoUpdate(m)
    InitializeHttpServer(m)
End Sub
Sub ty_InitializeWebView()
    InitializeHtmlWidget(m)
End Sub
Sub ty_EventLoop()
    while true
        msg = wait(0, m.msgPort)
        msg_type = type(msg)
        print "type(msg)=";msg_type
        handler = m.handlers[msg_type]
        if (handler <> invalid) then
            handler(msg)
        end if
    end while
End Sub
Sub ty_AddToEventLoopWithUserData(ifMessagePort as Object, event as String, userData as Object, callback as Object)
    ifMessagePort.SetPort(m.msgPort)
    userData.event = event
    userData.callback = callback
    ifMessagePort.SetUserData(userData)
    m.locks[event] = ifMessagePort
    m.handlers[event] = ty_priv_OnEventCallback
End Sub
Sub ty_AddToEventLoop(ifMessagePort as Object, event as String, callback as Object)
    m.AddToEventLoopWithUserData(ifMessagePort, event, {}, callback)
End Sub
Sub ty_priv_OnEventCallback(e as Object)
    user_data = e.GetUserData()
    if (type(user_data) = "roAssociativeArray") then
        event = user_data.event
        callback = user_data.callback
        if (type(event) = "roString" AND type(callback) = "roFunction") then
            result = callback(e)
            if (result <> false) then
                ty = GetGlobalAA().ty
                ty.locks[event] = invalid
                ty.handlers[event] = invalid
            end if
        end if
    end if
End Sub
Sub Main()
    port = CreateObject("roMessagePort")
    ty = NewTouchify(port)
    ty.InitializeSystem()
    ty.InitializeServices()
    ty.InitializeWebView()
    ty.EventLoop()
End Sub